/*
 * Written by Dawid Kurzyniec and released to the public domain, as explained
 * at http://creativecommons.org/licenses/publicdomain
 */

package edu.emory.mathcs.util.remote.server.impl;

import java.io.*;
import edu.emory.mathcs.util.remote.server.*;
import edu.emory.mathcs.util.remote.io.server.*;
import edu.emory.mathcs.util.remote.io.server.impl.*;

/**
 * Server-side remote handle to the native process.
 *
 * @see edu.emory.mathcs.util.remote.RemoteProcess
 *
 * @author Dawid Kurzyniec
 * @version 1.0
 */
public class RemoteProcessSrvImpl implements RemoteProcessSrv {
    final Process process;
    final RemoteInputStreamSrv srvin;
    final RemoteOutputStreamSrv srvout;
    final RemoteInputStreamSrv srverr;

    /**
     * Creates a new handle to the specified process.
     *
     * @param process the process to create handle to
     */
    public RemoteProcessSrvImpl(Process process) {
        if (process == null) {
            throw new NullPointerException("process");
        }
        this.process = process;
        this.srvin  = new RemoteInputStreamSrvImpl(process.getInputStream());
        this.srverr = new RemoteInputStreamSrvImpl(process.getErrorStream());
        this.srvout = new RemoteOutputStreamSrvImpl(process.getOutputStream());
    }

    public void destroy() {
        process.destroy();
    }

    public int waitFor() throws InterruptedException {
        return process.waitFor();
    }

    public int exitValue() {
        return process.exitValue();
    }

    public byte[] readFromIn(int maxbytes) throws IOException {
        return srvin.read(maxbytes);
    }
    public int availableInIn() throws IOException {
        return srvin.available();
    }
    public long skipInIn(long n) throws IOException {
        return srvin.skip(n);
    }
    public void closeIn() throws IOException {
        srvin.close();
    }
    public byte[] readFromErr(int maxbytes) throws IOException {
        return srverr.read(maxbytes);
    }
    public int availableInErr() throws IOException {
        return srverr.available();
    }
    public long skipInErr(long n) throws IOException {
        return srverr.skip(n);
    }
    public void closeErr() throws IOException {
        srverr.close();
    }
    public void writeToOut(byte[] buf) throws IOException {
        srvout.write(buf);
    }
    public void flushOut() throws IOException {
        srvout.flush();
    }
    public void closeOut() throws IOException {
        srvout.close();
    }
}
